// Matrixuhr-Mini, Scott-Falk Hühn, "main.cpp" ("matrixuhr-mini.ino")
// Hauptprogramm

// Versionsnummern und Texte
#define VERSION_NUMB "V1.10 * SFH"                // Versionsnummer
#define VERSION_DATE "2025-10-27"                 // Versionsdatum
#define SPI_CS1_PIN  15                           // SPI-CS1: Pin 15/D8, Chip-Select für oberes Display-Modul
#define SPI_CS2_PIN  16                           // SPI-CS2: Pin 16/D0, Chip-Select für unteres Display-Modul
                                                  // SPI-CLK: Pin 14/D5, SPI-MOSI: Pin 13/D7 (intern fest definiert)
#define ADC_PIN      A0                           // Analog-Eingang für Helligkeitssensor
#define WIFITOUT     5                            // Zeit in Sekunden für neuen Verbindungsversuch nach einer WLAN-Unterbrechung
#define SYNCTOUT     150                          // Zeit in Minuten bis zur Erkennung eines Sync-Ausfalls (1-254)
#define BOOTTOUT     25                           // Wartezeit in Zyklen von 40ms bis zum Reboot des ESP8266
#define MQTTTOUT     5                            // Zeit in Sekunden für neuen Verbindungsversuch nach einer MQTT-Unterbrechung
#define FILEDATA     16                           // maximale Anzahl der Zeilen im Dateipuffer
#define MAXBDAYS     100                          // maximale Anzahl der Einträge in der Geburtstagsliste
#define MAXBPAGE     25                           // maximale Anzahl der Listenelemente auf der Geburtstagsseite
#define MAXFILES     20                           // maximale Anzahl der Dateien im Dateisystem
#define SENCHARS     13                           // maximale Anzahl der Zeichen der über MQTT empfangenen Sensorwerte
#define JSONBUFF     1536                         // maximale Anzahl der Zeichen der über MQTT empfangenen JSON-Daten (Wetter- und Kraftstoffdaten)
#define TBUFFSIZE    100                          // maximale Anzahl der Zeichen im Text-Puffer (für Nachrichtenanzeige)
#define DATASIZE     150                          // maximale Anzahl der Zeichen im Datenfeld (für Datenanzeige)
#define PBUFFSIZE    1000                         // maximale Anzahl der Pixelspalten im Scroll-Puffer
#define TIMEZONE     TZ_Europe_Berlin             // Voreinstellung für die Zeitzone

// Bibliotheken einbinden
#include <Arduino.h>                              // Arduino-Basis-Funktionen
#include <SPI.h>                                  // SPI für Display-Ansteuerung
#include <LittleFS.h>                             // Dateisystem LittleFS
#include <ESP8266WiFi.h>                          // WLAN-Funktionen
#include <sntp.h>                                 // zusätzliche NTP-Funktionen
#include <coredecls.h>                            // Callback-Funktion für NTP
#include <TZ.h>                                   // Zeitzonen
#include <ESPAsyncWebServer.h>                    // Web-Server
#include <espMqttClient.h>                        // MQTT-Client
#include <ArduinoJson.h>                          // JSON-Bibliothek

// Dateinamen
#define wifi_filename "/wifi.txt"                 // Dateiname für WLAN-Daten (SSID und Passwort)
#define netw_filename "/network.txt"              // Dateiname für Netzwerk-Einstellungen
#define time_filename "/time.txt"                 // Dateiname für Zeit-Einstellungen
#define mqtt_filename "/mqtt.txt"                 // Dateiname für MQTT-Einstellungen
#define http_filename "/http.txt"                 // Dateiname für HTTP-Authentifizierung
#define bday_filename "/birthdays.txt"            // Dateiname für Geburtstagsliste
#define sens_filename "/sensors.txt"              // Dateiname für Sensor-Konfiguration
#define data_filename "/datadisp.txt"             // Dateiname für Datenausgabe-Konfiguration
#define disp_filename "/display.txt"              // Dateiname für Anzeige-Konfiguration

// Variablen
uint8_t matrix[2][32];                            // Matrix-Puffer, jeweils 1 Bereich für das oberes und unteres Display-Modul mit jeweils 4 x 8 Zeilenbytes
                                                  // Aufbau der einer Matrix:
                                                  // Byte-00 0...7 | Byte-01 0...7 | Byte-02 0...7 | Byte-03 0...7 | Zeile 0 (untere Zeile)
                                                  // Byte-04 0...7 | Byte-05 0...7 | Byte-06 0...7 | Byte-07 0...7 | Zeile 1
                                                  //      ...
                                                  // Byte-24 0...7 | Byte-25 0...7 | Byte-26 0...7 | Byte-27 0...7 | Zeile 6
                                                  // Byte-28 0...7 | Byte-29 0...7 | Byte-30 0...7 | Byte-31 0...7 | Zeile 7 (obere Zeile)
uint8_t dispmode;                                 // Anzeige-Modus (0-2)
                                                  // 0: normale Anzeige der Uhrzeit, keine Laufschrift aktiv
                                                  // 1: Laufschrift der Datenausgabe ist aktiv
                                                  // 2: Laufschrift einer Nachricht ist aktiv
uint8_t cur_matrix;                               // aktuell genutze Matrix für Daten- und Nachrichten-Ausgabe
uint8_t matrix_pos;                               // aktuelle X-Position bei Zeichenausgabe direkt in die Matrix
String wifi_ssid;                                 // WLAN-SSID (wird aus WLAN-Datei gelesen)
String wifi_pass;                                 // WLAN-Paswort (wird aus WLAN-Datei gelesen)
const char *wifi_ssid_p;                          // WLAN-SSID (wird aus WLAN-Datei gelesen)
const char *wifi_pass_p;                          // WLAN-Passwort (wird aus WLAN-Datei gelesen)
bool netw_enab;                                   // statische IP-Adresse aktivieren (wird aus Netzwerk-Datei gelesen)
String netw_addr;                                 // IP-Adresse (wird aus Netzwerk-Datei gelesen)
String netw_subn;                                 // Subnetz (wird aus Netzwerk-Datei gelesen)
String netw_gate;                                 // Gateway (wird aus Netzwerk-Datei gelesen)
String netw_dns1;                                 // DNS-Server (wird aus Netzwerk-Datei gelesen)
IPAddress IPaddr;                                 // IP-Adresse (wird aus Netzwerk-Datei gelesen)
IPAddress IPgate;                                 // Gateway (wird aus Netzwerk-Datei gelesen)
IPAddress IPsubn;                                 // Subnetzmaske (wird aus Netzwerk-Datei gelesen)
IPAddress IPdns1;                                 // DNS (wird aus Netzwerk-Datei gelesen)
bool mqtt_enab;                                   // MQTT aktivieren (wird aus MQTT-Datei gelesen)
String mqtt_addr;                                 // MQTT-Server-Adresse (wird aus MQTT-Datei gelesen)
String mqtt_port;                                 // MQTT-Server-Port (wird aus MQTT-Datei gelesen)
String mqtt_user;                                 // MQTT-Username (wird aus MQTT-Datei gelesen)
String mqtt_pass;                                 // MQTT-Passwort (wird aus MQTT-Datei gelesen)
String mqtt_ltop;                                 // MQTT LWT-Topic (wird aus MQTT-Datei gelesen)
String mqtt_utop;                                 // MQTT Uptime-Topic (wird aus MQTT-Datei gelesen)
String mqtt_rtop;                                 // MQTT RSSI-Topic (wird aus MQTT-Datei gelesen)
uint8_t mqtt_vali;                                // MQTT Werte-Gültigkeit (wird aus MQTT-Datei gelesen)
const char *mqtt_addr_p;                          // MQTT-Adresse (wird aus MQTT-Datei gelesen)
uint16_t mqtt_port_i;                             // MQTT-Server-Port (wird aus MQTT-Datei gelesen)
const char *mqtt_user_p;                          // MQTT-Username (wird aus MQTT-Datei gelesen)
const char *mqtt_pass_p;                          // MQTT-Passwort (wird aus MQTT-Datei gelesen)
const char *mqtt_ltop_p;                          // MQTT LWT-Topic (wird aus MQTT-Datei gelesen)
const char *mqtt_utop_p;                          // MQTT Uptime-Topic (wird aus MQTT-Datei gelesen)
const char *mqtt_rtop_p;                          // MQTT RSSI-Topic (wird aus MQTT-Datei gelesen)
String time_ntp1;                                 // NTP-Server 1 (wird aus Zeit-Datei gelesen)
String time_ntp2;                                 // NTP-Server 2 (wird aus Zeit-Datei gelesen)
String time_zone;                                 // Zeitzonen-Einstellungen (wird aus Zeit-Datei gelesen)
const char *time_ntp1_p;                          // NTP-Server 1 (wird aus Zeit-Datei gelesen)
const char *time_ntp2_p;                          // NTP-Server 2 (wird aus Zeit-Datei gelesen)
const char *time_zone_p;                          // Zeitzonen-Einstellungen (wird aus Zeit-Datei gelesen)
bool http_enab;                                   // HTTP-Authentifizierung aktiv (wird aus HTTP-Datei gelesen)
String http_user;                                 // HTTP-User-String (wird aus HTTP-Datei gelesen)
String http_pass;                                 // HTTP-Passwort-String (wird aus HTTP-Datei gelesen)
const char *http_user_p;                          // HTTP-Benutzer (wird aus HTTP-Datei gelesen)
const char *http_pass_p;                          // HTTP-Passwort (wird aus HTTP-Datei gelesen)
String birthdays[MAXBDAYS];                       // Datenfeld mit Geburtstagsdaten (wird aus Geburtstags-Datei gelesen)
String bdaylist;                                  // Geburtstagsliste für die Web-Seite
uint8_t bdaycount;                                // Zähler für Geburtstagsliste
uint8_t bdaynumb;                                 // eingegebene Eintragsnummer in der Geburtstagsliste
String bdaydate;                                  // eingegebenes Datum in der Geburtstagsliste
String bdayname;                                  // eingegebener Name in der Geburtstagsliste
String bdayadd;                                   // Zwischenspeicher für Klick auf "Hinzufügen"
String bdaydels;                                  // Zwischenspeicher für Klick auf "Löschen"
uint8_t bdaycurr;                                 // heute aktive Geburtstage
uint8_t bdaypage;                                 // aktuelle Web-Seite der Geburtstagsliste (0 = Seite 1, 1 = Seite 2, ...)
char sensorvals[8][SENCHARS];                     // über MQTT empfangene Sensorwerte (jeweils maximal SENCHARS Zeichen)
unsigned long sensortime[8];                      // Zeitstempel (Millis) der über MQTT empfangenen Sensorwerte
JsonDocument json;                                // JSON-Puffer für die Dekodierung der Wetter- und Kraftstoffdaten
char weatherjson[JSONBUFF];                       // über MQTT empfangene Wetterdaten
char fueljson[JSONBUFF];                          // über MQTT empfangene Kraftstoffdaten
unsigned long weathertime;                        // Zeitstempel (Millis) der über MQTT empfangenen Wetterdaten
unsigned long fueltime;                           // Zeitstempel (Millis) der über MQTT empfangenen Kraftstoffdaten
bool weatherflag;                                 // zeigt an, wenn neue Wetterdaten empfangen wurden (wenn true)
bool fuelflag;                                    // zeigt an, wenn neue Kraftstoffdaten empfangen wurden (wenn true)
String weatv;                                     // erste 12 Zeichen der empfangenen Wetterdaten (für Web-Seite)
String fuelv;                                     // erste 12 Zeichen der empfangenen Kraftstoffdaten (für Web-Seite)
String mesgv;                                     // erste 12 Zeichen der empfangenen Textnachricht (für Web-Seite)
char weathervals[13][15];                         // Datenfeld für die einzelnen Wetterinformationen:
                                                  //  0 - Temperatur in °C mit einer Nachkommastelle
                                                  //  1 - Temperatur in °C ohne Nachkommastelle
                                                  //  2 - Luftfeuchtigkeit in % ohne Nachkommastelle
                                                  //  3 - Luftdruck in hPa ohne Nachkommastelle
                                                  //  4 - Windgeschwindigkeit in m/s mit einer Nachkommastelle
                                                  //  5 - Windgeschwindigkeit in km/h ohne Nachkommastelle
                                                  //  6 - Regenmenge in mm/h mit einer Nachkommastelle
                                                  //  7 - Regenmenge in mm/h ohne Nachkommastelle
                                                  //  8 - Wolkendichte in % ohne Nachkommastelle
                                                  //  9 - Symbol für Windrichtungspfeil ($0 - $8)
                                                  // 10 - Wettersymbol ($A - $S)
                                                  // 11 - Wetterlage, Tabelleneintrag (0-54)
                                                  // 12 - Zeiten für Sonnenaufgang und Sonnenuntergang
uint16_t sunrise;                                 // Sonnenaufgangszeit (Minuten seit 0:00 Uhr)
uint16_t sunset;                                  // Sonnenuntergangszeit (Minuten seit 0:00 Uhr)
uint16_t curtime;                                 // aktuelle Zeit (Minuten seit 0:00 Uhr)
bool nightsymbol;                                 // Nachtmodus für Wettersymbol
bool nighttime;                                   // Nachtzeit für Display
char fuelvals[4][12];                             // Datenfeld für die einzelnen Kraftstoffpreise
bool dispconf;                                    // Anzeige-Konfiguration (true: 2 Display-Module verwenden) (wird aus Anzeige-Datei gelesen)
bool dispsecs;                                    // Anzeige der Uhrzeit mit Sekunden (wird aus Anzeige-Datei gelesen)
uint8_t dispintv;                                 // Datenausgabe-Intervall (0: aus, 1: 15s, 2: 20s, 3: 30s, 4: 60s, 5: 120s) (wird aus Anzeige-Datei gelesen)
uint8_t msg_count;                                // Nachrichten anzeigen (1: einnmal, 2: zweimal, 3: dreimal, 4: viermal, 5: fünfmal) (wird aus Anzeige-Datei gelesen)
String separator;                                 // dynamisches Trenn-Element (wird aus Anzeige-Datei gelesen)
uint8_t scrolltm1;                                // Scroll-Geschwindigkeit der Daten (0-11) (wird aus Anzeige-Datei gelesen)
uint8_t scrolltm2;                                // Scroll-Geschwindigkeit der Textnachricht (0-11) (wird aus Anzeige-Datei gelesen)
uint8_t night_stim;                               // Beginnzeit der Nacht (0-23) (wird aus Anzeige-Datei gelesen)
uint8_t night_etim;                               // Endezeit der Nacht (0-23) (wird aus Anzeige-Datei gelesen)
uint8_t save;                                     // Status der letzten Speicheraktion für die Anzeige auf der Web-Seite (0 = keine Anzeige, 1 = OK, 2 = Fehler)
bool result;                                      // Ergebnis der letzten Dateioperation
bool filestat;                                    // Dateisystem vorhanden (wenn true)
bool ntp_ok;                                      // NTP-Dienst wurde gestartet (wenn true)
bool mqtt_ok;                                     // MQTT-Dienst wurde gestartet (wenn true)
bool mqtt_actv;                                   // MQTT Aktive-Status, wird beim Systemstart von mqtt_enab übernommen
bool mqttstat;                                    // aktueller MQTT-Status (true: online)
bool time_change;                                 // Flag für die Steuerung des Doppelpunktes der Zeitanzeige; wird beim Sekundenwechsel gesetzt und nach einer halben Sekunde wieder gelöscht
bool reboot;                                      // Flag für Hinweis auf Neustart
uint8_t fwupstat;                                 // Firmware-Update-Status (0 = Update ist aktiv, 1 = Update war erfolgreich, 2 = Update war fehlerhaft)
unsigned long scrolltime;                         // Scroll-Zeit für Daten- und Nachrichtenausgabe
String filedata[FILEDATA];                        // Dateipuffer zum Lesen und Schreiben von Dateien
const char *ipval;                                // Zeiger auf aktuellen IP-Wert, wird zur Konvertierung aus Stringwert benötigt
IPAddress ipadd;                                  // Zwischenspeicher zur Plausibilitätsprüfung einer IP-Adresse
bool err_flag;                                    // Fehler-Flag, wird beid er Konvertierung aus Stringwert benötigt
String filenames[MAXFILES];                       // Dateinamen-Liste für Datei-Manager
String filesizes[MAXFILES];                       // Dateigrößen-Liste für Datei-Manager
char filedate[40];                                // Zwischenspeicher für Datei-Manager (Dateizeit)
String filelist;                                  // Dateiliste für die Web-Seite
uint8_t filecount;                                // Zähler für Datei-Manager
uint8_t filenumb;                                 // eingegebene Dateinummer im Datei-Manager
String filetemp;                                  // Zwischenspeicher für Datei-Manager
uint32_t fwcount;                                 // Bytezähler beim Firmware-Update
String sensorlist[11];                            // Datenfeld mit den MQTT-Topics der 8 freien Sensoren und 3 speziellen Informationen (wird aus Sensor-Datei gelesen)
const char *sensorlist_p[11];                     // Datenfeld mit den MQTT-Topics der 8 freien Sensoren und 3 speziellen Informationen (wird aus Sensor-Datei gelesen)
uint8_t sensordeci[8];                            // Datenfeld mit den Nachkommastellen der 8 freien Sensoren (0-3) (wird aus Sensor-Datei gelesen)
String datadisp;                                  // Datenfeld für die Datenausgabe-Konfiguration (wird aus Datenausgabe-Datei gelesen)
unsigned long ms40millis;                         // Zwischenspeicher für Millisekundenwert (für 40ms-Timeout-Zähler-Funktionen)
unsigned long lastmillis;                         // Zwischenspeicher für Millisekundenwert (für blinkenden Doppelpunkt)
unsigned long luptmillis;                         // Zwischenspeicher für Millisekundenwert (für die Ermittlung der Laufzeit)
unsigned long scrollmillis;                       // Zwischenspeicher für Millisekundenwert (für das Scrollen der Texte)
uint8_t lastsec;                                  // letzter gespeicherter Sekundenwert (für blinkenden Doppelpunkt und Timeout-Zähler-Funktionen)
uint8_t seccount;                                 // Sekundenzähler für Timeout-Funktionen (0-59)
uint8_t wifistat;                                 // aktueller WLAN-Status (0: normaler Verbindungsaufbau, 1: offline, 2: online)
uint8_t syncstat;                                 // Status der Zeit-Synchronsierung (0-2)
                                                  // 0 = Uhr ist nicht synchronisiert (Status nach Systemstart) -> Doppelpunkt in Uhrzeit erscheint als einzelner Punkt
                                                  // 1 = Uhr wurde mindestens 3 NTP-Intervalle nicht synchronisiert (Sync-Ausfall) -> Doppelpunkt in Uhrzeit erscheint als einzelner Punkt
                                                  // 2 = Uhr wurde innerhalb 3 NTP-Intervalle synchronisiert (Normalzustand) -> Doppelpunkt wird normal angezeigt
uint8_t wifitout;                                 // WLAN-Timeout-Zähler für Neuverbindung nach Unterbrechung, nach Ablauf der Zeit wird ein Verbindungsversuch gestartet; wird vom Hauptprogramm
                                                  // gesetzt und im Sekundentakt auf 0 gezählt, das Hauptprogramm setzt nach Bearbeitung den Wert auf 255 (Stopp)
uint8_t synctout;                                 // Synchronisierungs-Timeout-Zähler, nach Ablauf der Zeit wird der Synchronisierungsstatus auf Sync-Ausfall (syncstat = 1) gesetzt; wird vom Haupt-
                                                  // gesetzt und im Minutentakt auf 0 gezählt, das Hauptprogramm setzt nach Bearbeitung den Wert auf 255 (Stopp)
uint8_t boottout;                                 // Reboot-Timeout-Zähler, nach Ablauf wird der ESP8266 neu gestartet; wird vom Hauptprogramm gesetzt und im 40ms-Takt auf 0 gezählt, das Haupt-
                                                  // programm setzt nach Bearbeitung den Wert auf 255 (Stopp)
uint8_t mqtttout;                                 // MQTT-Timeout-Zähler für Neuverbindung nach Unterbrechung, nach Ablauf der Zeit wird ein Verbindungsversuch gestartet; wird vom Hauptprogramm
                                                  // gesetzt und im Sekundentakt auf 0 gezählt, das Hauptprogramm setzt nach Bearbeitung den Wert auf 255 (Stopp)
char uptime[8];                                   // formatierter String mit der Systemlaufzeit in Tagen mit 2 Nachkommastellen (für MQTT-Ausgabe)
char uptimex[15];                                 // formatierter String mit dynamischer Systemlaufzeit in Minuten, Stunden oder Tagen (für Web-Seite)
char strbuf[50];                                  // Stringpuffer für die Datenausgabe
char curdate[20];                                 // Zwischenspeicher für aktuelles Datum
String message;                                   // Speicher für die Textnachricht
bool messageflag;                                 // zeigt an, wenn eine neue Textnachricht empfangen wurde (wenn true)
int8_t rssi;                                      // RSSI-Wert in dB
const char *mqtt_lmsg_p;                          // MQTT LWT-Message (wird fest auf "offline" gesetzt)
const char *mqtt_smsg_p;                          // MQTT LWT-Start-Message (wird fest auf "online" gesetzt)
struct tm timestart;                              // Variable für Zeitinformationen zum Setzen der Uhrzeit nach Systemstart
time_t now;                                       // Variable für aktuelle Zeit
struct tm *timeinfo;                              // aktuelle Zeit in strukturierter Form
uint16_t year;                                    // Jahr
uint8_t month;                                    // Monat
uint8_t day;                                      // Tag
uint8_t wday;                                     // Wochentag
uint8_t hour;                                     // Stunde
uint8_t minute;                                   // Minute
uint8_t second;                                   // Sekunde
uint8_t hour_h;                                   // Stunden Zehner
uint8_t hour_l;                                   // Stunden Einer
uint8_t minute_h;                                 // Minuten Zehner
uint8_t minute_l;                                 // Minuten Einer
uint8_t second_h;                                 // Sekunden Zehner
uint8_t second_l;                                 // Sekunden Einer
long uptime_m;                                    // Systemlaufzeit in Minuten
uint8_t bright;                                   // aktueller Helligkeitswert
byte britab[32];                                  // Helligkeits-Tabelle zur Mittelwert-Berechnung
byte bripos = 0;                                  // aktuelle Schreibposition in der Helligkeits-Tabelle
uint8_t brightval;                                // aktueller berechneter Helligkeits-Mittelwert
uint8_t pixelbuff[PBUFFSIZE];                     // Pixelpuffer für die Scroll-Funktion, jedes Byte enthält eine Pixelspalte, Bit 0 ist das oberste Pixel
int16_t pbuff_wpos;                               // aktuelle Schreibposition im Pixelpuffer
int16_t pbuff_rpos;                               // aktuelle Leseposition im Pixelpuffer
int16_t pbuff_end;                                // Endposition des Textes im Pixelpuffer
uint8_t scr_count;                                // Zähler für die Ausgabe einer kompletten Nachricht
bool scrstop;                                     // Flag für das Ende der Laufschriftausgabe, nach dem Setzen werden noch 32 leere Pixelspalten ausgegeben
bool startup;                                     // Flag für Systemstart, sorgt für die einmalige Anzeige der IP-Adresse und des RSSI-Wertes
bool dataout;                                     // Flag für den Start der Datenausgabe
String serialstr;                                 // Puffer-String für empfangene serielle Daten
String serialcmd;                                 // Puffer-String für das empfangene Kommando
char serialchar;                                  // Zwischenspeicher für empfangenes serielles Zeichen
String newssid;                                   // neuer WLAN-SSID-String (wird seriell empfangen)
String newpass;                                   // neuer WLAN-Passwort-String (wird seriell empfangen)

WiFiClient espClient;                             // WLAN-Client einrichten
AsyncWebServer server(80);                        // Web-Server auf Port 80 einrichten
espMqttClient mqttClient;                         // MQTT-Client einrichten

#include "tables.h"
#include "functions.h"
#include "process.h"
#include "http.h"

void setup() { // ====================================================== Initialisierung ======================================================================

  Serial.begin(115200);                                               // USB-Kommunikation aktivieren
  Serial.println(F("\n\nSystemstart"));
  pinMode(SPI_CS1_PIN, OUTPUT);                                       // SPI-Select-Pin 1 aktivieren
  digitalWrite(SPI_CS1_PIN, HIGH);                                    // und auf High setzen
  pinMode(SPI_CS2_PIN, OUTPUT);                                       // SPI-Select-Pin 2 aktivieren
  digitalWrite(SPI_CS2_PIN, HIGH);                                    // und auf High setzen
  SPI.begin();                                                        // SPI aktivieren
  SPI.beginTransaction(SPISettings(4000000, MSBFIRST, SPI_MODE0));    // SPI-Parameter setzen, Taktfrequenz 4 MHz
  matrix_init();                                                      // Matrix-Displaymodule initialisieren
  matrix_clear(0);                                                    // Matrix 0 löschen
  matrix_clear(1);                                                    // Matrix 1 löschen
  matrix_send(0);                                                     // Matrix 0 ausgeben
  matrix_send(1);                                                     // Matrix 1 ausgeben

  if (!LittleFS.begin()) {                                            // wenn Dateisystem noch nicht vorhanden
    Serial.println(F("Dateisystem wird angelegt"));
    if (LittleFS.begin()) {                                           // Dateisystem anlegen
      filestat = true;                                                // Dateisystem ist ok
      Serial.println(F("Dateisystem wurde angelegt"));
    }
    else {                                                            // Dateisystem ist nicht ok
      Serial.println(F("Fehler beim Anlegen des Dateisystems"));
    }
  }
  else {                                                              // wenn Dateisystem vorhanden
    filestat = true;                                                  // Dateisystem ist ok
    Serial.println(F("Dateisystem ok"));
  }

  // ------------------------------------------------------------------- Einstellungen und Konfiguration aus Dateisystem lesen --------------------------------

  Serial.print(F("WLAN-Einstellungen lesen ..."));
  result = read_cfgfile(wifi_filename, 2);                            // WLAN-Zugangsdaten aus Dateisystem lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    wifi_ssid = filedata[0];                                          // SSID holen
    wifi_pass = filedata[1];                                          // Passwort holen
    if (wifi_ssid.length() == 0) wifi_ssid = "wifi_ssid";             // wenn keine SSID -> Dummy-Wert setzen
    if (wifi_pass.length() == 0) wifi_pass = "wifi_pass";             // wenn kein Passwort -> Dummy-Wert setzen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Lesen nicht erfolgreich
    wifi_ssid = "wifi_ssid";                                          // SSID-Dummy-Wert setzen
    wifi_pass = "wifi_pass";                                          // Passwort-Dummy-Wert setzen
    Serial.println(F(" Fehler"));
  }
  wifi_ssid_p = wifi_ssid.c_str();                                    // Zeiger auf SSID setzen
  wifi_pass_p = wifi_pass.c_str();                                    // Zeiger auf Paswort setzen

  Serial.print(F("Netzwerk-Einstellungen lesen ..."));
  result = read_cfgfile(netw_filename, 5);                            // Netzwerk-Einstellungen aus Dateisystem lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    netw_enab = (filedata[0].toInt() == 1)? true:false;               // statische IP-Adresse Aktivierung holen
    netw_addr = filedata[1];                                          // IP-Adresse holen
    netw_subn = filedata[2];                                          // Subnetzmaske holen
    netw_gate = filedata[3];                                          // Gateway holen
    netw_dns1 = filedata[4];                                          // DSN holen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Fehler beim Lesen
    netw_enab = false;                                                // Voreinstellungen setzen
    netw_addr = "192.168.1.230";
    netw_subn = "255.255.255.0";
    netw_gate = "192.168.1.1";
    netw_dns1 = "192.168.1.1";
    Serial.println(F(" Fehler"));
  }
  err_flag = false;                                                   // Fehler-Flag löschen
  ipval = netw_addr.c_str();                                          // Zeiger setzen
  if (!IPaddr.fromString(ipval)) err_flag = true;                     // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
  ipval = netw_subn.c_str();                                          // Zeiger setzen
  if (!IPsubn.fromString(ipval)) err_flag = true;                     // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
  ipval = netw_gate.c_str();                                          // Zeiger setzen
  if (!IPgate.fromString(ipval)) err_flag = true;                     // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
  ipval = netw_dns1.c_str();                                          // Zeiger setzen
  if (!IPdns1.fromString(ipval)) err_flag = true;                     // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
  if (err_flag) netw_enab = false;                                    // wenn Konvertierung fehlgeschlagen ist -> statische IP-Adresse deaktivieren

  Serial.print(F("MQTT-Einstellungen lesen ..."));
  result = read_cfgfile(mqtt_filename, 9);                            // MQTT-Einstellungen aus Dateisystem lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    mqtt_enab = (filedata[0].toInt() == 1)? true:false;               // MQTT Aktivierung holen
    mqtt_addr = filedata[1];                                          // MQTT Server-Adresse holen
    mqtt_port = filedata[2];                                          // MQTT Server-Port holen
    mqtt_user = filedata[3];                                          // MQTT Username holen
    mqtt_pass = filedata[4];                                          // MQTT Passwort holen
    mqtt_ltop = filedata[5];                                          // MQTT LWT-Topic holen
    mqtt_utop = filedata[6];                                          // MQTT Uptime-Topic holen
    mqtt_rtop = filedata[7];                                          // MQTT RSSI-Topic holen
    mqtt_vali = filedata[8].toInt();                                  // MQTT Werte-Gültigkeit holen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Fehler beim Lesen
    mqtt_enab = false;                                                // Voreinstellungen setzen
    mqtt_addr = "192.168.1.222";
    mqtt_port = "1883";
    mqtt_user = "";
    mqtt_pass = "";
    mqtt_ltop = "Haus/Technik/MatrixuhrMini-1/LWT";
    mqtt_utop = "Haus/Technik/MatrixuhrMini-1/uptime";
    mqtt_rtop = "Haus/Technik/MatrixuhrMini-1/rssi";
    mqtt_vali = 10;
    Serial.println(F(" Fehler"));
  }
  if (mqtt_addr.length() == 0 || mqtt_port.length() == 0) mqtt_enab = false; // wenn keine MQTT-Server-Adresse oder kein MQTT-Server-Port -> MQTT deaktivieren
  if (mqtt_enab) {                                                    // wenn MQTT aktiviert
    mqtt_addr_p = mqtt_addr.c_str();                                  // Zeiger auf MQTT-Server-Adresse setzen
    mqtt_port_i = mqtt_port.toInt();                                  // MQTT-Port in Zahl umwandeln
    if (mqtt_user.length() > 0) mqtt_user_p = mqtt_user.c_str();      // wenn MQTT-Username vorhanden -> Zeiger setzen
    else mqtt_user_p = nullptr;                                       // sonst Nullzeiger setzen
    if (mqtt_pass.length() > 0) mqtt_pass_p = mqtt_pass.c_str();      // wenn MQTT-Passwort vorhanden -> Zeiger setzen
    else mqtt_pass_p = nullptr;                                       // sonst Nullzeiger setzen
    if (mqtt_ltop.length() > 0) mqtt_ltop_p = mqtt_ltop.c_str();      // wenn MQTT LWT-Topic vorhanden -> Zeiger setzen
    else mqtt_ltop_p = nullptr;                                       // sonst Nullzeiger setzen
    if (mqtt_utop.length() > 0) mqtt_utop_p = mqtt_utop.c_str();      // wenn MQTT Uptime-Topic vorhanden -> Zeiger setzen
    else mqtt_utop_p = nullptr;                                       // sonst Nullzeiger setzen
    if (mqtt_rtop.length() > 0) mqtt_rtop_p = mqtt_rtop.c_str();      // wenn MQTT RSSI-Topic vorhanden -> Zeiger setzen
    else mqtt_rtop_p = nullptr;                                       // sonst Nullzeiger setzen
    if (mqtt_vali < 2 || mqtt_vali > 60) mqtt_vali = 10;              // wenn Werte-Gültigkeit außerhalb des Bereiches -> Voreinstellung setzen
  }
  mqtt_actv = mqtt_enab;                                              // MQTT-Aktivierung wird auf aktuellen Status übertragen

  Serial.print(F("Zeit-Einstellungen lesen ..."));
  result = read_cfgfile(time_filename, 3);                            // Zeit-Einstellungen aus Dateisystem lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    time_ntp1 = filedata[0];                                          // NTP-Server 1 holen
    time_ntp2 = filedata[1];                                          // NTP-Server 2 holen
    time_zone = filedata[2];                                          // Zeitzonen-Einstellung holen
    if (time_ntp1.length() == 0) time_ntp1 = "de.pool.ntp.org";       // wenn kein NTP-Server 1 -> Voreinstellung setzen
    if (time_zone.length() == 0) time_zone = String(TIMEZONE);        // wenn keine Zeitzonen-Einstellung -> Voreinstellung setzen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Lesen nicht erfolgreich
    time_ntp1 = "de.pool.ntp.org";                                    // Voreinstellungen setzen
    time_ntp2 = "ptbtime1.ptb.de";
    time_zone = String(TIMEZONE);
    Serial.println(F(" Fehler"));
  }
  time_ntp1_p = time_ntp1.c_str();                                    // Zeiger auf NTP-Server 1 setzen
  if (time_ntp2.length() > 0) time_ntp2_p = time_ntp2.c_str();        // wenn NTP2-Server definiert -> Zeiger auf NTP-Server 2 setzen
  else time_ntp2_p = nullptr;                                         // sonst Nullzeiger setzen
  time_zone_p = time_zone.c_str();                                    // Zeiger auf Zeitzonen-Einstellung setzen

  Serial.print(F("HTTP-Authentifizierung lesen ..."));
  result = read_cfgfile(http_filename, 3);                            // HTTP-Authentifizierung aus Dateisystem lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    http_enab = (filedata[0].toInt() == 1)? true:false;               // HTTP-Aktivierung holen
    http_user = filedata[1];                                          // HTTP-Username holen
    http_pass = filedata[2];                                          // HTTP-Passwort holen
    if (http_user.length() == 0) http_user = "admin";                 // wenn kein HTTP-Username -> Dummy-Wert setzen
    if (http_pass.length() == 0) http_pass = "admin";                 // wenn kein HTTP-Passwort -> Dummy-Wert setzen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Fehler beim Lesen
    http_enab = false;                                                // Voreinstellungen setzen
    http_user = "admin";
    http_pass = "admin";
    Serial.println(F(" Fehler"));
  }
  http_user_p = http_user.c_str();                                    // Zeiger auf HTTP-Username setzen
  http_pass_p = http_pass.c_str();                                    // Zeiger auf HTTP-Passwort setzen

  Serial.print(F("Geburtstage lesen ..."));
  result = read_bdayfile();                                           // Geburtstagsdatei lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    bdaysort();                                                       // Geburtstags-Datenfeld sortieren und Anzahl ermitteln
    Serial.print(F(" ok, Eintraege: "));
    Serial.println(bdaycount);
  }
  else Serial.println(F(" Fehler"));

  Serial.print(F("Sensoren lesen ..."));
  result = read_cfgfile(sens_filename, 11);                           // Sensorliste aus Dateisystem lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    for (uint8_t i = 0; i < 8; i ++) {                                // 8 Einträge bearbeiten (freie Sensoren)
      if (filedata[i].length() > 0) {                                 // wenn MQTT-Topic vorhanden
        sensorlist[i] = filedata[i].substring(0, filedata[i].length() - 2);      // Eintrag holen und MQTT Topic filtern
        sensordeci[i] = filedata[i].substring(filedata[i].length() - 1).toInt(); // Eintrag holen und Nachkommastelle filtern
        if (sensordeci[i] > 3) sensordeci[i] = 0;                     // wenn Bereich überschritten -> Voreinstellung setzen
      }
    }
    for (uint8_t i = 8; i < 11; i ++) sensorlist[i] = filedata[i];    // 3 spezielle Einträge holen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Fehler beim Lesen
    for (uint8_t i = 0; i < 8; i ++) {                                // Voreinstellungen setzen
      sensorlist[i] = "";
      sensordeci[i] = 0;
    }
    sensorlist[8] = "Haus/Info/Wetter/aktuell";
    sensorlist[9] = "Haus/Info/Tanken/Preise";
    sensorlist[10] = "Haus/Info/Nachrichten";
    Serial.println(F(" Fehler"));
  }
  for (uint8_t i = 0; i < 11; i ++) {                                 // 11 Zeiger setzen
    if (sensorlist[i].length() > 0) sensorlist_p[i] = sensorlist[i].c_str(); // wenn MQTT-Topic vorhanden -> Zeiger setzen
    else sensorlist_p[i] = nullptr;                                   // sonst Nullzeiger setzen
  }

  Serial.print(F("Datenausgabe lesen ..."));
  result  = read_cfgfile(data_filename, 1);                           // Datenausgabe-Konfiguration aus Dateisystem lesen
  if (result) {                                                       // wenn Lesen erfolgreich
    datadisp = filedata[0];                                           // 1 Eintrag holen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Fehler beim Lesen
    datadisp = "";                                                    // Voreinstellung setzen
    Serial.println(F(" Fehler"));
  }

  Serial.print(F("Anzeige lesen ..."));
  result = read_cfgfile(disp_filename, 9);                            // Anzeige-Konfiguration aus Dateisystem lesen

  if (result) {                                                       // wenn Lesen erfolgreich
    dispconf = (filedata[0].toInt() == 1)? true:false;                // Anzeige-Konfiguration holen
    dispsecs = (filedata[1].toInt() == 1)? true:false;                // Sekundenanzeige holen
    dispintv = filedata[2].toInt();                                   // Anzeige-Intervall holen
    msg_count = filedata[3].toInt();                                  // Nachrichten anzeigen holen
    separator = filedata[4];                                          // dynamisches Trenn-Element holen
    scrolltm1 = filedata[5].toInt();                                  // Scroll-Geschwindigkeit 1 holen
    scrolltm2 = filedata[6].toInt();                                  // Scroll-Geschwindigkeit 2 holen
    night_stim = filedata[7].toInt();                                 // Nacht-Beginnzeit holen
    night_etim = filedata[8].toInt();                                 // Nacht-Endezeit holen
    if (dispintv > 5) dispintv = 3;                                   // wenn Bereich überschritten -> Voreinstellung setzen
    if (msg_count < 1 || msg_count > 5) msg_count = 3;                // wenn Bereich überschritten -> Voreinstellung setzen
    if (scrolltm1 > 11) scrolltm1 = 6;                                // wenn Bereich überschritten -> Voreinstellung setzen
    if (scrolltm2 > 11) scrolltm2 = 6;                                // wenn Bereich überschritten -> Voreinstellung setzen
    if (night_stim > 23) night_stim = 0;                              // wenn Bereich überschritten -> Voreinstellung setzen
    if (night_etim > 23) night_etim = 0;                              // wenn Bereich überschritten -> Voreinstellung setzen
    Serial.println(F(" ok"));
  }
  else {                                                              // wenn Fehler beim Lesen
    dispconf = false;                                                 // Voreinstellungen setzen
    dispsecs = false;
    dispintv = 3;
    msg_count = 3;
    separator = "xxx";
    scrolltm1 = 6;
    scrolltm2 = 6;
    night_stim = 0;
    night_etim = 0;
    Serial.println(F(" Fehler"));
  }

  // ------------------------------------------------------------------- Hardware und Variablen initialisieren ------------------------------------------------

  WiFi.mode(WIFI_STA);                                                // WLAN-Station (Client-Modus)
  if (netw_enab) WiFi.config(IPaddr, IPgate, IPsubn, IPdns1);         // wenn feste IP-Adresse aktiviert -> IP-Daten einstellen
  WiFi.setAutoReconnect(true);                                        // WLAN-Verbindung bei Trennung automatisch wiederherstellen
  mqtt_lmsg_p = "offline";                                            // MQTT LWT-Message setzen
  mqtt_smsg_p = "online";                                             // MQTT LWT-Start-Message setzen
  timestart.tm_year = 2025 - 1900;                                    // Jahr voreinstellen
  timestart.tm_mon = 0;                                               // Monat voreinstellen (Januar)
  timestart.tm_mday = 1;                                              // Tag voreinstellen
  timestart.tm_hour = 0;                                              // Stunde voreinstellen
  timestart.tm_min = 0;                                               // Minute voreinstellen
  timestart.tm_sec = 0;                                               // Sekunde voreinstellen
  timestart.tm_isdst = 0;                                             // Normalzeit voreinstellen
  time_t t = mktime(&timestart);                                      // Zeit konvertieren
  struct timeval now = { .tv_sec = t };                               // Zeit als Epoch-Wert (Sekunden seit 1970) verwenden
  settimeofday(&now, NULL);                                           // neue Zeit setzen
  settimeofday_cb(time_is_set);                                       // Callback-Funktion für Synchronisierung einrichten
  uptime_strings();                                                   // formatierte Systemlaufzeit erstellen

  wifistat = 0;                                                       // WLAN-Verbindung herstellen
  syncstat = 0;                                                       // Uhr ist nicht synchronisiert
  mqttstat = false;                                                   // MQTT-Verbindung herstellen
  startup = true;                                                     // Systemstart setzen
  wifitout = 255;                                                     // WLAN-Timeout-Zähler stoppen
  boottout = 255;                                                     // Neustart-Timeout-Zähler stoppen
  synctout = 255;                                                     // Synchronisierungs-Timeout-Zähler stoppen
  mqtttout = 255;                                                     // MQTT-Timeout-Zähler stoppen
  lastsec = 60;                                                       // letzten gespeicherten Sekundenwert setzen
  lastmillis = millis();                                              // aktuellen Millisekundenwert zwischenspeichern (für Hauptschleife initialisieren)
  luptmillis = millis();                                              // aktuellen Millisekundenwert zwischenspeichern (für Emittlung der Laufzeit)

  // ------------------------------------------------------------------- Web-Server-Definitionen --------------------------------------------------------------

  server.on("/", HTTP_GET, [](AsyncWebServerRequest *request) {       // Web-Server, Ausgabe der Hauptseite (Status) definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", stat_html, processor);          // Seite stat_html ausgeben und Variablen bearbeiten
  });

  server.on("/", HTTP_POST, [](AsyncWebServerRequest *request) {      // Web-Server, übergebene Informationen der Hauptseite (Status)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", stat_html, processor);          // Seite stat_html neu ausgeben und Variablen berabeiten
  });

  server.on("/favicon.ico", [](AsyncWebServerRequest *request) {      // Web-Server, Ausgabe des Icon (favicon.ico)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    request->send_P(200, "image/png", favicon, sizeof(favicon));      // favicon.png (32x32 Pixel) ausgeben
  });

  server.on("/mesg", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /mesg (Nachricht anzeigen)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "message") message = p->value();             // wenn Parameter message übergeben wurde -> Wert als String speichern
      }
    }
    if (message[0] > 0) {                                             // wenn Nachricht vorhanden
      process_msg();                                                  // Nachricht ausgeben
      save = 1;                                                       // OK-Meldung
    }
    request->send_P(200, "text/html", stat_html, processor);          // Seite stat_html neu ausgeben und Variablen berabeiten
  });

  server.on("/bday", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /bday definieren (Geburtstage)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    bdaypage = 0;                                                     // Geburtstagsseite 1 anzeigen
    bdaylister();                                                     // Geburtstagsliste für Web-Seite erstellen
    request->send_P(200, "text/html", bday_html, processor);          // Seite bday_html ausgeben und Variablen bearbeiten
  });

  server.on("/bday", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /bday (Geburtstage)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "bday_date") bdaydate = p->value();          // wenn Parameter bday_date übergeben wurde -> Wert speichern (Datum)
        if (p->name() == "bday_name") bdayname = p->value();          // wenn Parameter bday_name übergeben wurde -> Wert speichern (Name)
      }
    }
    if (bdayname.length() > 0 && bdaydate.length() > 0) {             // wenn Name und Datum eingegeben
      for (uint8_t i = 0; i < MAXBDAYS; i ++) {                       // erstes freies Feld suchen
        if (birthdays[i].length() == 0) {                             // wenn freies Feld gefunden
          birthdays[i] = bdaydate.substring(0, 4) + bdaydate.substring(5, 7) + bdaydate.substring(8) + " " + bdayname; // Datum umwandeln und Name ergänzen
          break;                                                      // Schleife abbrechen
        }
      }
      bdaypage = 0;                                                   // Geburtstagsseite 1 anzeigen
      bdaysort();                                                     // Geburtstags-Datenfeld sortieren und Anzahl ermitteln
      bdaylister();                                                   // Geburtstagsliste für Web-Seite erstellen
      result = write_bdayfile();                                      // Geburtstagdatei schreiben
      if (result) save = 1;                                           // wenn Speichern erfolgreich -> OK-Meldung
      else save = 2;                                                  // wenn Speichern nicht erfolgreich -> Fehler
    }
    request->send_P(200, "text/html", bday_html, processor);          // Seite bday_html neu ausgeben und Variablen berabeiten
  });

  server.on("/bdel", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /bdel (Geburtstage/Eintrag löschen)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    bdaynumb = 0;                                                     // Eingabe zunächst auf ungültigen Wert setzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "bday_deln") bdaynumb = p->value().toInt();  // wenn Parameter bday_deln übergeben wurde -> Wert speichern (Nummer)
      }
    }
    if (bdaynumb > 0 && bdaynumb <= bdaycount) {                      // wenn gültige Eintragnummer
      birthdays[bdaynumb - 1] = "";                                   // Feldelement löschen
      bdaypage = 0;                                                   // Geburtstagsseite 1 anzeigen
      bdaysort();                                                     // Geburtstags-Datenfeld sortieren und Anzahl ermitteln
      bdaylister();                                                   // Geburtstagsliste für Web-Seite erstellen
      result = write_bdayfile();                                      // Geburtstagdatei schreiben
      if (result) save = 1;                                           // wenn Speichern erfolgreich -> OK-Meldung
      else save = 2;                                                  // wenn Speichern nicht erfolgreich -> Fehler
    }
    request->send_P(200, "text/html", bday_html, processor);          // Seite bday_html neu ausgeben und Variablen berabeiten
  });

  server.on("/bdpg", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /bdpg (Geburtstage/Seite)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    bdaypage ++;                                                      // auf nächste Seite setzen
    if ((bdaypage * MAXBPAGE) >= bdaycount) bdaypage = 0;             // wenn oberes Ende erreicht -> wieder auf Seite 0 setzen
    bdaylister();                                                     // Geburtstagsliste für Web-Seite erstellen
    request->send_P(200, "text/html", bday_html, processor);          // Seite bday_html neu ausgeben und Variablen bearbeiten
  });

  server.on("/sens", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /sens definieren (Sensoren)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", sens_html, processor);          // Seite sens_html ausgeben und Variablen bearbeiten
  });

  server.on("/sens", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /sens (Sensoren)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "sena") filedata[0] = p->value();            // wenn Parameter sena übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor A)
        if (p->name() == "senb") filedata[1] = p->value();            // wenn Parameter senb übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor B)
        if (p->name() == "senc") filedata[2] = p->value();            // wenn Parameter senc übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor C)
        if (p->name() == "send") filedata[3] = p->value();            // wenn Parameter send übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor D)
        if (p->name() == "sene") filedata[4] = p->value();            // wenn Parameter sena übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor E)
        if (p->name() == "senf") filedata[5] = p->value();            // wenn Parameter senb übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor F)
        if (p->name() == "seng") filedata[6] = p->value();            // wenn Parameter senc übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor G)
        if (p->name() == "senh") filedata[7] = p->value();            // wenn Parameter send übergeben wurde -> Wert als String zwischenspeichern (Topic Sensor H)
        if (p->name() == "weat") filedata[8] = p->value();            // wenn Parameter weat übergeben wurde -> Wert als String zwischenspeichern (Topic Wetterdaten)
        if (p->name() == "fuel") filedata[9] = p->value();            // wenn Parameter fuel übergeben wurde -> Wert als String zwischenspeichern (Topic Kraftstoffdaten)
        if (p->name() == "mesg") filedata[10] = p->value();           // wenn Parameter mesg übergeben wurde -> Wert als String zwischenspeichern (Topic Textnachricht)
        if (p->name() == "seda") sensordeci[0] = p->value().toInt();  // wenn Parameter deca übergeben wurde -> Sensor A Nachkommastelle als Zahl speichern
        if (p->name() == "sedb") sensordeci[1] = p->value().toInt();  // wenn Parameter decb übergeben wurde -> Sensor B Nachkommastelle als Zahl speichern
        if (p->name() == "sedc") sensordeci[2] = p->value().toInt();  // wenn Parameter decc übergeben wurde -> Sensor C Nachkommastelle als Zahl speichern
        if (p->name() == "sedd") sensordeci[3] = p->value().toInt();  // wenn Parameter decd übergeben wurde -> Sensor D Nachkommastelle als Zahl speichern
        if (p->name() == "sede") sensordeci[4] = p->value().toInt();  // wenn Parameter dece übergeben wurde -> Sensor E Nachkommastelle als Zahl speichern
        if (p->name() == "sedf") sensordeci[5] = p->value().toInt();  // wenn Parameter decf übergeben wurde -> Sensor F Nachkommastelle als Zahl speichern
        if (p->name() == "sedg") sensordeci[6] = p->value().toInt();  // wenn Parameter decg übergeben wurde -> Sensor G Nachkommastelle als Zahl speichern
        if (p->name() == "sedh") sensordeci[7] = p->value().toInt();  // wenn Parameter dech übergeben wurde -> Sensor H Nachkommastelle als Zahl speichern
      }
    }
    if (mqttstat && mqttClient.connected()) {                         // wenn MQTT-Status online und Server-Verbindung aktiv
      for (uint8_t i = 0; i < 11; i ++) {                             // 11 Datenfelder bearbeiten
        if (sensorlist_p[i]) mqttClient.unsubscribe(sensorlist_p[i]); // wenn Datenfeld ein MQTT-Topic enthält -> Topic abbestellen
      }
    }
    for (uint8_t i = 0; i < 11; i ++) {                               // 11 Datenfelder bearbeiten
      sensorlist[i] = filedata[i];                                    // zwischengespeichertes MQTT-Topic in Sensorliste kopieren
      if (sensorlist[i].length() > 0) sensorlist_p[i] = sensorlist[i].c_str(); // wenn MQTT-Topic vorhanden -> Zeiger setzen
      else sensorlist_p[i] = nullptr;                                 // sonst Nullzeiger setzen
    }
    if (mqttstat && mqttClient.connected()) {                         // wenn MQTT-Status online und Server-Verbindung aktiv
      for (uint8_t i = 0; i < 11; i ++) {                             // 11 Datenfelder bearbeiten
        if (sensorlist_p[i]) mqttClient.subscribe(sensorlist_p[i], 0); // wenn Datenfeld ein MQTT-Topic enthält -> Topic abonnieren
      }
    }
    for (uint8_t i = 0; i < 8; i ++) {                                // 8 Datenfelder bearbeiten (freie Sensoren)
      if (filedata[i].length() > 0) filedata[i] += " " + String(sensordeci[i]); // wenn Topic vorhanden -> Nachkommastelle ergänzen
    }
    result = write_cfgfile(sens_filename, 11);                        // Sensorliste ins Dateisystem schreiben
    if (result) save = 1;                                             // wenn Speichern erfolgreich -> OK-Meldung
    else save = 2;                                                    // wenn Speichern nicht erfolgreich -> Fehler
    request->send_P(200, "text/html", sens_html, processor);          // Seite sens_html neu ausgeben und Variablen berabeiten
  });

  server.on("/seac", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /seac (Sensoren/Aktualisieren)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", sens_html, processor);          // Seite sens_html neu ausgeben und Variablen bearbeiten
  });

  server.on("/data", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /data definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", data_html, processor);          // Seite sena_html ausgeben und Variablen bearbeiten
  });

  server.on("/data", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /data
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "data") datadisp = p->value();               // wenn Parameter data übergeben wurde -> Wert als String speichern (Datenfeld 1)
      }
    }
    filedata[0] = datadisp;                                           // Datenfeld in Dateipuffer schreiben
    result = write_cfgfile(data_filename, 1);                         // Datenausgabe-Konfiguration ins Dateisystem schreiben
    if (result) save = 1;                                             // wenn Speichern erfolgreich -> OK-Meldung
    else save = 2;                                                    // wenn Speichern nicht erfolgreich -> Fehler
    request->send_P(200, "text/html", data_html, processor);          // Seite sena_html neu ausgeben und Variablen berabeiten
  });

  server.on("/daac", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /daac (Datenausgabe/Aktualisieren)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", data_html, processor);          // Seite data_html neu ausgeben und Variablen berabeiten
  });

  server.on("/disp", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /disp definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", disp_html, processor);          // Seite disp_html ausgeben und Variablen bearbeiten
  });

  server.on("/disp", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /disp
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    if (params > 0) {                                                 // wenn mindestens ein Parameter übergeben wurde
      dispconf = false;                                               // Anzeige-Konfiguration zunächst auf 1 Displaymodul setzen
      dispsecs = false;                                               // Sekunden-Anzeige zunächst ausschalten
    }
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "disp_conf") dispconf = true;                // wenn Parameter disp_conf übergeben wurde -> Wert speichern (Anzeige-Modus)
        if (p->name() == "disp_secs") dispsecs = true;                // wenn Parameter disp_secs übergeben wurde -> Wert speichern (Sekunden-Anzeige)
        if (p->name() == "disp_intv") dispintv = p->value().toInt();  // wenn Parameter disp_intv übergeben wurde -> Wert speichern (Anzeige-Intervall)
        if (dispintv > 5) dispintv = 3;                               // wenn Wert außerhalb des Bereiches -> auf Default-Wert setzen
        if (p->name() == "disp_mesg") msg_count = p->value().toInt(); // wenn Parameter disp_mesg übergeben wurde -> Wert speichern (Nachrichten anzeigen)
        if (msg_count < 1 || msg_count > 5) msg_count = 3;            // wenn Wert außerhalb des Bereiches -> auf Default-Wert setzen
        if (p->name() == "disp_sepa") separator = p->value();         // wenn Parameter disp_sepa übergeben wurde -> Wert speichern (dynamisches Trenn-Element)
        if (p->name() == "disp_scr1") scrolltm1 = p->value().toInt(); // wenn Parameter disp_scr1 übergeben wurde -> Wert speichern (Scroll-Geschwindigkeit 1)
        if (scrolltm1 > 11) scrolltm1 = 6;                            // wenn Wert außerhalb des Bereiches -> auf Default-Wert setzen
        if (p->name() == "disp_scr2") scrolltm2 = p->value().toInt(); // wenn Parameter disp_scr1 übergeben wurde -> Wert speichern (Scroll-Geschwindigkeit 2)
        if (scrolltm2 > 11) scrolltm2 = 6;                            // wenn Wert außerhalb des Bereiches -> auf Default-Wert setzen
        if (p->name() == "nigh_stim") night_stim = p->value().toInt(); // wenn Parameter nigh_stim übergeben wurde -> Wert als Zahl speichern (Nacht-Beginnzeit)
        if (night_stim > 23) night_stim = 0;                          // wenn Bereich überschritten -> Voreinstellung setzen
        if (p->name() == "nigh_etim") night_etim = p->value().toInt(); // wenn Parameter nigh_etim übergeben wurde -> Wert als Zahl speichern (Nacht-Endzeit)
        if (night_etim > 23) night_etim = 0;                          // wenn Bereich überschritten -> Voreinstellung setzen
      }
    }
    if (!dispconf) {                                                  // wenn nur 1 Displaymodul verwendet wird
      matrix_clear(1);                                                // untere Matrix löschen
      matrix_send(1);                                                 // untere Matrix aktualisieren
    }
    filedata[0] = dispconf? '1':'0';                                  // Zeile 1 - Anzeige-Modus
    filedata[1] = dispsecs? '1':'0';                                  // Zeile 2 - Sekunden-Anzeige
    filedata[2] = String(dispintv);                                   // Zeile 3 - Anzeige-Intervall
    filedata[3] = String(msg_count);                                  // Zeile 4 - Nachrichten anzeigen
    filedata[4] = separator;                                          // Zeile 5 - dynamisches Trenn-Element
    filedata[5] = String(scrolltm1);                                  // Zeile 6 - Scroll-Geschwindigkeit 1
    filedata[6] = String(scrolltm2);                                  // Zeile 7 - Scroll-Geschwindigkeit 2
    filedata[7] = String(night_stim);                                 // Zeile 8 - Nacht-Beginnzeit
    filedata[8] = String(night_etim);                                 // Zeile 9 - Nacht-Endezeit
    result = write_cfgfile(disp_filename, 9);                         // Anzeige-Konfiguration ins Dateisystem schreiben
    if (result) save = 1;                                             // wenn Speichern erfolgreich -> OK-Meldung
    else save = 2;                                                    // wenn Speichern nicht erfolgreich -> Fehler
    request->send_P(200, "text/html", disp_html, processor);          // Seite disp_html neu ausgeben und Variablen berabeiten
  });

  server.on("/wifi", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /wifi definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", wifi_html, processor);          // Seite wifi_html ausgeben und Variablen bearbeiten
  });

  server.on("/wifi", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /wifi
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "wifi_user") {                               // wenn Parameter wifi_user übergeben wurde
          wifi_ssid = p->value();                                     // Wert als String speichern
          wifi_ssid_p = wifi_ssid.c_str();                            // und als WLAN-SSID verwenden
        }
        if (p->name() == "wifi_pass") {                               // wenn Parameter wifi_pass übergeben wurde
          wifi_pass = p->value();                                     // Wert als String speichern
          wifi_pass_p = wifi_pass.c_str();                            // und als WLAN-Passwort verwenden
        }
      }
    }
    filedata[0] = wifi_ssid;                                          // Zeile 1 - WLAN-SSID
    filedata[1] = wifi_pass;                                          // Zeile 2 - WLAN-Passwort
    result = write_cfgfile(wifi_filename, 2);                         // WLAN-Zugangsdaten ins Dateisystem schreiben
    if (result) { save = 1; reboot = true; }                          // wenn Speichern erfolgreich -> OK-Meldung und Neustart anfordern
    else save = 2;                                                    // wenn Speichern nicht erfolgreich -> Fehler
    request->send_P(200, "text/html", wifi_html, processor);          // Seite wifi_html neu ausgeben und Variablen berabeiten
  });

  server.on("/netw", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /netw definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", netw_html, processor);          // Seite netw_html ausgeben und Variablen bearbeiten
  });

  server.on("/netw", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /netw
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    err_flag = false;                                                 // Fehler-Flag löschen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    if (params > 0) netw_enab = false;                                // wenn mindestens ein Parameter übergeben wurde -> statische IP-Adresse zunächst deaktivieren
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "netw_enab") netw_enab = true;               // wenn Parameter netw_enab übergeben wurde -> Wert speichern (statische IP-Adresse aktivieren)
        if (p->name() == "netw_addr") {                               // wenn Parameter netw_addr übergeben wurde
          netw_addr = p->value();                                     // Wert als String speichern
          ipval = netw_addr.c_str();                                  // Zeiger setzen
          if (!IPaddr.fromString(ipval)) err_flag = true;             // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
          else netw_addr = IPaddr.toString();                         // Ergebnis in String zurückwandeln
        }
        if (p->name() == "netw_subn") {                               // wenn Parameter netw_subn übergeben wurde
          netw_subn = p->value();                                     // Wert als String speichern
          ipval = netw_subn.c_str();                                  // Zeiger setzen
          if (!IPsubn.fromString(ipval)) err_flag = true;             // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
          else netw_subn = IPsubn.toString();                         // Ergebnis in String zurückwandeln
        }
        if (p->name() == "netw_gate") {                               // wenn Parameter netw_gate übergeben wurde
          netw_gate = p->value();                                     // Wert als String speichern
          ipval = netw_gate.c_str();                                  // Zeiger setzen
          if (!IPgate.fromString(ipval)) err_flag = true;             // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
          else netw_gate = IPgate.toString();                         // Ergebnis in String zurückwandeln
        }
        if (p->name() == "netw_dns1") {                               // wenn Parameter netw_dns1 übergeben wurde
          netw_dns1 = p->value();                                     // Wert als String speichern
          ipval = netw_dns1.c_str();                                  // Zeiger setzen
          if (!IPdns1.fromString(ipval)) err_flag = true;             // ins IP-Format konvertieren, Fehler setzen wenn nicht erfolgreich
          else netw_dns1 = IPdns1.toString();                         // Ergebnis in String zurückwandeln
        }
      }
    }
    if (err_flag) netw_enab = false;                                  // wenn Konvertierung fehlgeschlagen ist -> statische IP-Adresse deaktivieren
    if (!err_flag) {                                                  // wenn kein Fehler-Flag gesetzt (IP-Werte sind plausibel)
      filedata[0] = netw_enab? '1':'0';                               // Zeile 1 - statische IP-Adresse Aktivierung, in Ziffer wandeln
      filedata[1] = netw_addr;                                        // Zeile 2 - IP-Adresse
      filedata[2] = netw_subn;                                        // Zeile 3 - Subnetzmaske
      filedata[3] = netw_gate;                                        // Zeile 4 - Gateway
      filedata[4] = netw_dns1;                                        // Zeile 5 - DNS
      result = write_cfgfile(netw_filename, 5);                       // IP-Einstellungen ins Dateisystem schreiben
      if (result) {                                                   // wenn Speichern erfolgreich
        save = 1;                                                     // OK-Meldung
        if (netw_enab) reboot = true;                                 // wenn statische IP-Adresse aktiviert -> Neustart anfordern
      }
      else save = 2;                                                  // wenn Speichern nicht erfolgreich -> Fehler
    }
    else save = 2;                                                    // wenn Fehler-Flag gesetzt (IP-Werte sind nicht plausibel) -> Fehler
    request->send_P(200, "text/html", netw_html, processor);          // Seite netw_html neu ausgeben und Variablen berabeiten
  });

  server.on("/mqtt", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /mqtt definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", mqtt_html, processor);          // Seite mqtt_html ausgeben und Variablen bearbeiten
  });

  server.on("/mqtt", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /mqtt
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    err_flag = false;                                                 // Fehler-Flag löschen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    if (params > 0) mqtt_enab = false;                                // wenn mindestens ein Parameter übergeben wurde -> MQTT-Aktivierung zunächst löschen
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "mqtt_enab") mqtt_enab = true;               // wenn Parameter mqtt_enab übergeben wurde -> MQTT aktivieren
        if (p->name() == "mqtt_addr") mqtt_addr = p->value();         // wenn Parameter mqtt_addr übergeben wurde -> Wert als String speichern
        if (mqtt_addr.length() == 0) err_flag = true;                 // wenn keine MQTT-Server-Addresse -> Fehler
        if (p->name() == "mqtt_port") mqtt_port = p->value();         // wenn Parameter mqtt_port übergeben wurde -> Wert als String speichern
        if (mqtt_port.length() == 0) err_flag = true;                 // wenn kein MQTT-Server-Port -> Fehler
        if (p->name() == "mqtt_user") mqtt_user = p->value();         // wenn Parameter mqtt_user übergeben wurde -> Wert als String speichern
        if (p->name() == "mqtt_pass") mqtt_pass = p->value();         // wenn Parameter mqtt_pass übergeben wurde -> Wert als String speichern
        if (p->name() == "mqtt_ltop") mqtt_ltop = p->value();         // wenn Parameter mqtt_ltop übergeben wurde -> Wert als String speichern
        if (p->name() == "mqtt_utop") mqtt_utop = p->value();         // wenn Parameter mqtt_utop übergeben wurde -> Wert als String speichern
        if (p->name() == "mqtt_rtop") mqtt_rtop = p->value();         // wenn Parameter mqtt_rtop übergeben wurde -> Wert als String speichern
        if (p->name() == "mqtt_vali") mqtt_vali = p->value().toInt(); // wenn Parameter mqtt_vald übergeben wurde -> Wert als Zahl speichern
        if (mqtt_vali < 2 || mqtt_vali > 60) mqtt_vali = 10;          // wenn Werte-Gültigkeit außerhalb des Bereiches -> Voreinstellung setzen
      }
    }
    if (err_flag) mqtt_enab = false;                                  // wenn fehlerhafte Werte -> MQTT deaktivieren
    if (mqtt_enab) {                                                  // wenn MQTT aktiviert
      mqtt_addr_p = mqtt_addr.c_str();                                // Zeiger auf MQTT-Adresse setzen
      mqtt_port_i = mqtt_port.toInt();                                // MQTT-Port in Zahl umwandeln
      if (mqtt_user.length() > 0) mqtt_user_p = mqtt_user.c_str();    // wenn MQTT-Username vorhanden -> Zeiger setzen
      else mqtt_user_p = nullptr;                                     // sonst Nullzeiger setzen
      if (mqtt_pass.length() > 0) mqtt_pass_p = mqtt_pass.c_str();    // wenn MQTT-Passwort vorhanden -> Zeiger setzen
      else mqtt_pass_p = nullptr;                                     // sonst Nullzeiger setzen
      if (mqtt_ltop.length() > 0) mqtt_ltop_p = mqtt_ltop.c_str();    // wenn MQTT LWT-Topic vorhanden -> Zeiger setzen
      else mqtt_ltop_p = nullptr;                                     // sonst Nullzeiger setzen
      if (mqtt_utop.length() > 0) mqtt_utop_p = mqtt_utop.c_str();    // wenn MQTT Uptime-Topic vorhanden -> Zeiger setzen
      else mqtt_utop_p = nullptr;                                     // sonst Nullzeiger setzen
      if (mqtt_rtop.length() > 0) mqtt_rtop_p = mqtt_rtop.c_str();    // wenn MQTT RSSI-Topic vorhanden -> Zeiger setzen
      else mqtt_rtop_p = nullptr;                                     // sonst Nullzeiger setzen
    }
    if (!err_flag) {                                                  // wenn kein Fehler-Flag gesetzt
      filedata[0] = mqtt_enab? '1':'0';                               // Zeile 1 - MQTT Aktivierung
      filedata[1] = mqtt_addr;                                        // Zeile 2 - MQTT Server-Adresse
      filedata[2] = mqtt_port;                                        // Zeile 3 - MQTT Port
      filedata[3] = mqtt_user;                                        // Zeile 4 - MQTT Username
      filedata[4] = mqtt_pass;                                        // Zeile 5 - MQTT Passwort
      filedata[5] = mqtt_ltop;                                        // Zeile 6 - MQTT LWT-Topic
      filedata[6] = mqtt_utop;                                        // Zeile 7 - MQTT Uptime-Topic
      filedata[7] = mqtt_rtop;                                        // Zeile 8 - MQTT RSSI-Topic
      filedata[8] = String(mqtt_vali);                                // Zeile 9 - MQTT Werte-Gültigkeit
      result = write_cfgfile(mqtt_filename, 9);                       // MQTT-Einstellungen ins Dateisystem schreiben
      if (result) {                                                   // wenn Speichern erfolgreich
        save = 1;                                                     // OK-Meldung
        if (mqtt_enab) reboot = true;                                 // wenn MQTT aktiviert -> Neustart anfordern
      }
      else save = 2;                                                  // wenn Speichern nicht erfolgreich -> Fehler
    }
    else save = 2;                                                    // wenn Fehler-Flag gesetzt -> Fehler
    request->send_P(200, "text/html", mqtt_html, processor);          // Seite mqtt_html neu ausgeben und Variablen berabeiten
  });

  server.on("/time", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /time definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", time_html, processor);          // Seite time_html ausgeben und Variablen bearbeiten
  });

  server.on("/time", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /time
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    err_flag = false;                                                 // Fehler-Flag löschen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "time_ntp1") {                               // wenn Parameter time_ntp1 übergeben wurde
          time_ntp1 = p->value();                                     // Wert als String speichern
          if (time_ntp1.length() == 0) err_flag = 1;                  // wenn NTP-Server 1 nicht angegeben -> Fehler
        }
        if (p->name() == "time_ntp2") {                               // wenn Parameter time_ntp2 übergeben wurde
          time_ntp2 = p->value();                                     // Wert als String speichern
        }
        if (p->name() == "time_zone") {                               // wenn Parameter time_zone übergeben wurde
          time_zone = p->value();                                     // Wert als String speichern
          if (time_zone.length() == 0) err_flag = 1;                  // wenn Zeitzonen-Information nicht angegeben -> Fehler
        }
      }
    }

    if (!err_flag) {                                                  // wenn kein Fehler
      filedata[0] = time_ntp1;                                        // Zeile 1 - NTP-Server 1
      filedata[1] = time_ntp2;                                        // Zeile 2 - NTP-Server 2
      filedata[2] = time_zone;                                        // Zeile 3 - Zeitzonen-Information
      result = write_cfgfile(time_filename, 3);                       // Zeit-Einstellungen ins Dateisystem schreiben
      if (result) { save = 1; reboot = true; }                        // wenn Speichern erfolgreich -> OK-Meldung und Neustart anfordern
      else save = 2;                                                  // wenn Speichern nicht erfolgreich -> Fehler
    }
    else save = 2;                                                    // wenn Fehler-Flag gesetzt -> Fehler
    request->send_P(200, "text/html", time_html, processor);          // Seite time_html neu ausgeben und Variablen berabeiten
  });

  server.on("/http", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /http definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", http_html, processor);          // Seite http_html ausgeben und Variablen bearbeiten
  });

  server.on("/http", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /http
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    if (params > 0) http_enab = false;                                // wenn mindestens ein Parameter übergeben wurde -> HTTP-Aktivierung zunächst löschen
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "http_enab") http_enab = true;               // wenn Parameter http_enab übergeben wurde -> Wert speichern (HTTP-Aktivierung)
        if (p->name() == "http_user") {                               // wenn Parameter http_user übergeben wurde
          http_user = p->value();                                     // Wert als String speichern
          http_user_p = http_user.c_str();                            // und als HTTP-Username verwenden
        }
        if (p->name() == "http_pass") {                               // wenn Parameter http_pass übergeben wurde
          http_pass = p->value();                                     // Wert als String speichern
          http_pass_p = http_pass.c_str();                            // und als HTTP-Passwort verwenden
        }
      }
    }
    filedata[0] = http_enab? '1':'0';                                 // Zeile 1 - HTTP-Aktivierung
    filedata[1] = http_user;                                          // Zeile 2 - HTTP-Username
    filedata[2] = http_pass;                                          // Zeile 3 - HTTP-Passwort
    result = write_cfgfile(http_filename, 3);                         // HTTP-Login-Einstellungen ins Dateisystem schreiben
    if (result) save = 1;                                             // wenn Speichern erfolgreich -> OK-Meldung
    else save = 2;                                                    // wenn Speichern nicht erfolgreich -> Fehler
    request->send_P(200, "text/html", http_html, processor);          // Seite http_html neu ausgeben und Variablen berabeiten
  });

  server.on("/file", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Ausgabe der Unterseite /file definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    filedir();                                                        // Dateiliste des Hauptverzeichnisses erstellen
    request->send_P(200, "text/html", file_html, processor);          // Seite file_html ausgeben und Variablen bearbeiten
  });

  server.on("/file", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /file (Dateimanager, Download)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    filenumb = 0;                                                     // Eingabe zunächst auf ungültigen Wert setzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "file_numb") filenumb = p->value().toInt();  // wenn Parameter file_numb übergeben wurde -> Wert speichern (Dateinummer)
      }
    }
    if (filenumb > 0 && filenumb <= filecount) {                      // wenn gültige Dateinummer
      filetemp = "/" + filenames[filenumb - 1];                       // Dateiname aus Liste holen
      request->send(LittleFS, filetemp, String(), true);              // Download starten
    }
    else {                                                            // wenn ungültige Dateinummer
      request->send_P(200, "text/html", file_html, processor);        // Seite file_html neu ausgeben und Variablen berabeiten
    }
  });

  server.on("/fdel", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /fdel (Dateimanager, Löschen)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    filenumb = 0;                                                     // Eingabe zunächst auf ungültigen Wert setzen
    int params = request->params();                                   // Anzahl der Parameter ermitteln
    for (int i = 0; i < params; i ++) {                               // Parameter nacheinander bearbeiten
      const AsyncWebParameter* p = request->getParam(i);              // Zeiger auf Parameter setzen
      if (p->isPost()) {                                              // wenn Post-Methode verwendet wird
        if (p->name() == "file_numb") filenumb = p->value().toInt();  // wenn Parameter file_numb übergeben wurde -> Wert speichern (Dateinummer)
      }
    }
    if (filenumb > 0 && filenumb <= filecount) {                      // wenn gültige Dateinummer
      filetemp = "/" + filenames[filenumb - 1];                       // Dateiname aus Liste holen
      result = false;                                                 // Ergebnis zunächst löschen
      if (filestat) LittleFS.remove(filetemp);                        // wenn Dateisystem vorhanden -> Datei löschen
      filedir();                                                      // Dateiliste des Hauptverzeichnisses erstellen
    }
    request->send_P(200, "text/html", file_html, processor);          // Seite file_html neu ausgeben und Variablen berabeiten
  });

  server.on("/fupl", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, Upload-Steuerung definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
  },[](AsyncWebServerRequest *request, String filename, size_t index, uint8_t *data, size_t len, bool final) {
    if (!index) {                                                     // wenn Upload beginnt
      request->_tempFile = LittleFS.open("/" + filename, "w");        // neue Datei erstellen
    }
    if (len) {                                                        // wenn Daten empfangen wurden
      request->_tempFile.write(data, len);                            // in neue Datei schreiben
    }
    if (final) {                                                      // wenn Upload beendet
      request->_tempFile.close();                                     // Datei schließen
      reboot = true;                                                  // Neustart anfordern
      request->redirect("/file");                                     // Datei-Manager-Seite neu ausgeben
    }
  });

  server.on("/fwup", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Unterseite /fwup definieren
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    fwcount = 0;                                                      // Update-Bytezähler löschen
    fwupstat = 0;                                                     // Firmware-Update-Status auf Aktiv setzen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->send_P(200, "text/html", fwup_html, processor);          // Seite fwup_html ausgeben und Variablen bearbeiten
  });

  server.on("/fwup", HTTP_POST, [](AsyncWebServerRequest *request) {  // Web-Server, übergebene Informationen der Unterseite /fwup
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    if (Update.hasError()) {                                          // wenn Fehler
      save = 2;                                                       // Fehler-Status setzen
      fwupstat = 2;                                                   // Firmware-Update-Status auf Fehler setzen
    }
    else {                                                            // wenn erfolgreich
      if (fwcount > 0) {                                              // wenn Daten empfangen wurden
        save = 1;                                                     // OK-Meldung
        fwupstat = 1;                                                 // Firmware-Update-Status auf Ok setzen
        reboot = true;                                                // Neustart anfordern
      }
    }
    AsyncWebServerResponse *response = request->beginResponse_P(200, "text/html", fwup_html, processor); // Seite fwup_html neu ausgeben und Variablen bearbeiten
    response->addHeader("Connection", "close");                       // Verbindung beenden
    request->send(response);                                          // Daten senden
  },[](AsyncWebServerRequest *request, String filename, size_t index, uint8_t *data, size_t len, bool final){
    if (!index) {                                                     // wenn Upload beginnt
      Serial.printf("Firmware-Update: %s\n", filename.c_str());       // Start protokollieren
      fwupstat = 0;                                                   // Firmware-Update-Status auf Aktiv setzen
      Update.runAsync(true);
      uint32_t maxfree = (ESP.getFreeSketchSpace() - 0x1000) & 0xFFFFF000; // freien Speicher in Flash ermitteln
      if (maxfree > 1048576) maxfree = 1048576;                       // wenn mehr als 1 MB -> auf 1 MB begrenzen
      if (!Update.begin(maxfree, U_FLASH))                            // wenn Update-Start fehlgeschlagen
        Update.printError(Serial);                                    // Fehlermeldung protokollieren
    }
    if (!Update.hasError()) {                                         // wenn Upload bisher fehlerfrei
      fwcount = index + len;                                          // Update-Bytezähler aktualisieren
      if (Update.write(data, len) != len) {                           // wenn Fehler beim Schreiben der Daten
        Update.printError(Serial);                                    // Fehlermeldung protokollieren
      }
    }
    if (final) {                                                      // wenn Upload beendet
      if (Update.end(true)) {                                         // wenn Upload erfolgreich
        if (fwcount > 0) fwupstat = 1;                                // wenn Daten empfangen wurden -> Firmware-Update-Status auf Ok setzen
        Serial.printf("Firmware-Update ok: %u Bytes\n", index + len); // Upload-Ende protokollieren
      }
      else {                                                          // wenn Upload nicht erfolgreich
        Update.printError(Serial);                                    // Fehlermeldung protokollieren
        fwupstat = 2;                                                 // Firmware-Update-Status auf Fehler setzen
      }
    }
  });

  server.on("/boot", HTTP_GET, [](AsyncWebServerRequest *request) {   // Web-Server, Ausgabe der Hauptseite (Status) definieren (wird von /boot umgeleitet)
    if (http_enab) if (!request->authenticate(http_user_p, http_pass_p)) return request->requestAuthentication(); // gegebenenfalls Login durchführen
    save = 0;                                                         // Speicherstatus zurücksetzen
    request->redirect("/");                                           // zur Hauptseite wechseln
    boottout = BOOTTOUT;                                              // Neustart-Timer setzen
  });

  server.onNotFound(notFound);                                        // wenn Seite nicht gefunden wird
}

void loop() { // ======================================================= Hauptprogrammschleife ================================================================

  int8_t i;                                                           // Zählervariable
  time(&now);                                                         // aktuelle Zeit holen
  timeinfo = localtime(&now);                                         // aktuelle Lokalzeit ermitteln
  year = timeinfo->tm_year + 1900;                                    // 4-stelliges Jahr ermitteln
  month = timeinfo->tm_mon + 1;                                       // Monat ermitteln (1-12)
  day = timeinfo->tm_mday;                                            // Tag ermitteln (1-31)
  wday = timeinfo->tm_wday + 1;                                       // Wochentag ermitteln (1-7, 1 = Sonntag)
  hour = timeinfo->tm_hour;                                           // Stunde ermitteln (0-23)
  minute = timeinfo->tm_min;                                          // Minute ermitteln (0-59)
  second = timeinfo->tm_sec;                                          // Sekunde ermitteln (0-59)
  sprintf(curdate, "%02d.%02d.%04d", day, month, year);               // aktuelles Datum für Web-Server generieren
  mqttClient.loop();                                                  // MQTT verarbeiten

  if (millis() - ms40millis >= 40) {                                  // wenn 40ms vergangen
    ms40millis = millis();                                            // aktuellen Millisekundenwert zwischenspeichern
    timer40ms();                                                      // Timeout-Zähler-Funktion bearbeiten
  }
  if (lastsec != second) {                                            // wenn sich die Sekunde geändert hat
    lastmillis = millis();                                            // aktuellen Millisekundenwert zwischenspeichern
    lastsec = second;                                                 // neuen Sekundenwert zwischenspeichern
    timer1sec();                                                      // Timeout-Zähler 1s bearbeiten
    if (nighttime) {                                                  // wenn Nachtzeit
      if (!startup && dispmode < 2) matrix_off();                     // wenn keine Startphase und keine Nachricht -> Matrix ausschalten
    }
    else matrix_reinit();                                             // wenn keine Nachtzeit -> Display reinitialiseren (Störungen entfernen)

    if (fwcount > 0) {                                                // wenn Firmware-Update aktiv
      time_change = false;                                            // Löschen des Doppelpunktes ausschalten
      scrstop = true;                                                 // Laufschrift beenden
      dispmode = 0;                                                   // normale Anzeige
      matrix_clear(1);                                                // untere Matrix löschen
      matrix_send(1);                                                 // untere Matrix aktualisieren
      matrix_clear(0);                                                // obere Matrix löschen
      switch (fwupstat) {                                             // Firmware-Update-Status auswerten
        case 0:                                                       // wenn Firmware-Update aktiv
          matrix_pos = 2;
          matrix_str(0, 12);                                          // Update-Text auf oberer Matrix ausgeben
          break;
        case 1:                                                       // wenn Firmware-Update Ok
          matrix_pos = 11;
          matrix_str(0, 13);                                          // Ok-Text auf oberer Matrix ausgeben
          break;
        case 2:                                                       // wenn Firmware-Update fehlerhaft
          matrix_pos = 2;
          matrix_str(0, 14);                                          // Fehler-Text auf oberer Matrix ausgeben
          break;
        default: break;
      }
      matrix_send(0);                                                 // obere Matrix aktualisieren
    }
    else {                                                            // wenn kein Firmware-Update aktiv
      time_change = true;                                             // Flag zum Löschen des Doppelpunktes setzen
      if (dispconf || (!dispconf &&  dispmode == 0)) {                // wenn 2 Display-Module oder nur ein Display-Modul ohne aktive Laufschrift
        matrix_time(0, true);                                         // Uhrzeit mit Doppelpunkt ausgeben
        matrix_send(0);                                               // Matrix aktualisieren
      }
    }

    dataout = false;                                                  // Datenausgabe-Flag löschen
    if (dispmode == 0) {                                              // wenn keine Laufschrift aktiv
      switch (dispintv) {                                             // Anzeigeintervall prüfen
        case 0:                                                       // wenn Intervall aus
          dataout = true;                                             // Datenausgabe sofort starten
          break;
        case 1:                                                       // wenn Anzeigeintervall = 15 Sekunden
          if (second % 15 == 0) dataout = true;                       // wenn Sekunde 0, 15, 30 oder 45 -> Datenausgabe starten
          break;
        case 2:                                                       // wenn Anzeigeintervall = 20 Sekunden
          if (second % 20 == 0) dataout = true;                       // wenn Sekunde 0, 20 oder 40 -> Datenausgabe starten
          break;
        case 3:                                                       // wenn Anzeigeintervall = 30 Sekunden
          if (second % 30 == 0) dataout = true;                       // wenn Sekunde 0 oder 30 -> Datenausgabe starten
          break;
        case 4:                                                       // wenn Anzeigeintervall = 60 Sekunden
          if (second == 0) dataout = true;                            // wenn Sekunde 0 -> Datenausgabe starten
          break;
        case 5:                                                       // wenn Anzeigeintervall = 120 Sekunden
          if (minute % 2 == 0 && second == 0) dataout = true;         // wenn gerade Minute und Sekunde 0 -> Datenausgabe starten
          break;
        default: break;                                               // wenn ungültiges Anzeigeintervall -> keine Aktion
      }
      if (dataout) process_data();                                    // wenn Datenausgabe-Flag -> Datenausgabe starten
    }
  }

  if (lastmillis + 500 <= millis()) {                                 // wenn zweite Sekundenhälfte
    if (time_change) {                                                // wenn Flag zum Löschen des Doppelpunktes gesetzt
      time_change = false;                                            // Flag wieder löschen
      if (dispconf || (!dispconf &&  dispmode == 0)) {                // wenn 2 Display-Module oder nur ein Display-Modul ohne aktive Laufschrift
        matrix_time(0, false);                                        // Uhrzeit ohne Doppelpunkt ausgeben
        matrix_send(0);                                               // Matrix aktualisieren
      }
    }
  }

  if (millis() - scrollmillis >= 64 - scrolltime * 4) {               // wenn Scrollzeit abgelaufen (0-11 wird umgerechnet in 64-20 Millisekunden-Zyklen)
    scrollmillis = millis();                                          // aktuellen Millisekundenwert zwischenspeichern
    scrolltime = scrolltm1;                                           // Scroll-Geschwindigkeit für Datenausgaben verwenden
    if (dispmode == 2) scrolltime = scrolltm2;                        // wenn Nachrichtenanzeige -> Scroll-Geschwindigkeit für Nachricht verwenden
    if (dispmode > 0) {                                               // wenn Laufschriftanzeige aktiv
      if (pbuff_rpos < pbuff_end) {                                   // wenn Ende im Pixelpuffer noch nicht erreicht
        cur_matrix = 0;                                               // zunächst Matrix 0 verwenden
        if (dispconf) cur_matrix = 1;                                 // wenn zwei Matrix-Module vorhanden -> untere Matrix verwenden
        matrix_shift(cur_matrix);                                     // aktuelle Matrix verschieben
        matrix_send(cur_matrix);                                      // aktuelle Matrix aktualisieren
      }
      pbuff_rpos ++;                                                  // Zeiger auf nächste Leseposition im Pixelpuffer
      if (pbuff_rpos == 0 && dispmode > 0 && scrstop) dispmode = 0;   // wenn Anfang des Pixelpuffers erreicht und Laufschrift-Ende -> zur normalen Anzeige wechseln
      if (pbuff_rpos >= pbuff_end) {                                  // wenn Ende im Pixelpuffer erreicht
        pbuff_rpos = -32;                                             // auf 32 Pixelspalten vor den Anfang setzen
        if (dispmode == 2) {                                          // wenn Nachrichtenanzeige aktiv
          scr_count ++;                                               // Nachrichtenausgabe-Zähler erhöhen
          if (scr_count >= msg_count) scrstop = true;                 // wenn Ausgabe-Anzahl erreicht -> Laufschrift beenden
          if (startup) {                                              // wenn Systemstart
            startup = false;                                          // Systemstart-Flag löschen
            scrstop = true;                                           // Laufschrift beenden
          }
        }
        if (dispmode == 1) {                                          // wenn Datenausgabe aktiv
          scrstop = true;                                             // Laufschrift beenden
        }
      }
    }
  }

  if (weatherflag) {                                                  // wenn neue Wetterdaten empfangen wurden
    decode_weather();                                                 // Wetterdaten dekodieren
    weatherflag = false;                                              // Flag wieder löschen
  }
  if (fuelflag) {                                                     // wenn neue Kraftstoffdaten empfangen wurden
    decode_fuel();                                                    // Kraftstoffdaten dekodieren
    fuelflag = false;                                                 // Flag wieder löschen
  }
  if (messageflag) {                                                  // wenn eine Textnachricht empfangen wurde
    process_msg();                                                    // Textnachricht bearbeiten
    messageflag = false;                                              // Flag wieder löschen
  }

// ------------------------------------------------------------------- Timeout-Zähler und Status auswerten --------------------------------------------------

  if (synctout == 0) {                                                // wenn Synchronisierungs-Timeout-Zähler abgelaufen
    synctout = 255;                                                   // Timeout-Zähler stoppen
    syncstat = 1;                                                     // Synchronisierungs-Status auf Sync-Ausfall setzen
  }

  if (wifitout == 0) {                                                // wenn WLAN-Timeout-Zähler abgelaufen
    wifitout = 255;                                                   // Timeout-Zähler stoppen
    if ((wifistat == 1) && (WiFi.status() != WL_CONNECTED)) {         // wenn WLAN im Offline-Zustand
      Serial.println(F("WLAN-Verbindung wiederherstellen"));
      WiFi.reconnect();                                               // WLAN-Verbindung wiederherstellen
    }
  }

  if (boottout == 0) {                                                // wenn Reboot-Timeout-Zähler abgelaufen
    Serial.println(F("System-Neustart"));
    if (mqtt_actv) mqttClient.disconnect();                           // wenn MQTT aktiviert -> Verbindung trennen
    WiFi.disconnect();                                                // WLAN-Verbindung trennen
    scrstop = true;                                                   // Laufschrift beenden
    dispmode = 0;                                                     // normale Anzeige
    matrix_clear(1);                                                  // untere Matrix löschen
    matrix_send(1);                                                   // untere Matrix aktualisieren
    matrix_clear(0);                                                  // obere Matrix löschen
    matrix_pos = 2;
    matrix_str(0, 15);                                                // Reboot-Text auf oberer Matrix ausgeben
    matrix_send(0);                                                   // obere Matrix aktualisieren
    delay(100);                                                       // kurze Wartezeit
    ESP.restart();                                                    // System-Neustart
  }

  if (millis() - luptmillis >= 60000) {                               // wenn mindestens eine Minute vergangen ist
    if (millis() >= luptmillis) luptmillis += 60000;                  // wenn Millisekundenwert normal -> Schwellwert auf nächste Minute setzen
    else luptmillis += 60000;                                         // sonst Schwellwert neu setzen (bei Überlauf des Millisekundenwertes)
    uptime_m ++;                                                      // Systemlaufzeit erhöhen
    uptime_strings();                                                 // formatierte Systemlaufzeit erstellen
    rssi = WiFi.RSSI();                                               // WLAN-Empfangssignal speichern
    if (mqttstat && mqttClient.connected()) {                         // wenn MQTT-Verbindung aktiv
      if (mqtt_utop_p) mqttClient.publish(mqtt_utop_p, 0, false, uptime);               // wenn MQTT-Topic für Systemlaufzeit definiert -> senden
      if (mqtt_rtop_p) mqttClient.publish(mqtt_rtop_p, 0, false, String(rssi).c_str()); // wenn MQTT-Topic für WLAN-Empfangssignal definiert -> senden
    }
  }

  // ------------------------------------------------------------------- WLAN-Status und MQTT-Status auswerten und steuern ------------------------------------

  if (wifistat == 0) {                                                // wenn Verbindungsaufbau gestartet werden soll
    wifistat = 1;                                                     // neuen Status "offline" setzen
    if (WiFi.status() != WL_CONNECTED) {                              // wenn aktuell keine WLAN-Verbindung
      if (((uint8_t) wifi_ssid_p[0] > 0) && ((uint8_t) wifi_pass_p[0] > 0)) { // wenn SSID und Passwort vorhanden
        WiFi.begin(wifi_ssid_p, wifi_pass_p);                                 // Verbindungsaufbau starten
        Serial.println(F("WLAN-Verbindungsaufbau"));
      }
    }
  }
  if ((wifistat == 1) && (WiFi.status() == WL_CONNECTED)) {           // wenn WLAN von offline in den Online-Zustand wechselt
    wifistat = 2;                                                     // neuen Status "online" setzen
    server.begin();                                                   // Server-Dienst starten
    Serial.println(F("WLAN-Verbindung hergestellt"));
    message = F("IP: ");                                              // Nachricht mit IP-Daten beginnen
    message += WiFi.localIP().toString();                             // IP-Adresse ergänzen
    rssi = WiFi.RSSI();                                               // RSSI-Wert ermitteln
    if (rssi < 0) {                                                   // wenn sinnvoller RSSI-Wert
      message += F("  RSSI: ");                                       // Zwischentext ergänzen
      message += rssi;                                                // RSSI-Wert ergänzen
      message += F(" dBm");                                           // Einheit ergänzen
    }
    Serial.println(message);
    if (startup) process_msg();                                       // wenn Systemstart -> Nachricht auf Display ausgeben
  }

  if ((wifistat == 2) && (WiFi.status() != WL_CONNECTED)) {           // wenn WLAN von online in den Offline-Zustand wechselt
    wifistat = 1;                                                     // neuen Status "offline" setzen
    Serial.println(F("WLAN-Verbindung unterbrochen"));
    wifitout = WIFITOUT;                                              // WLAN-Timeout-Zähler starten und neue Verbindung versuchen
  }

  if (wifistat == 2) {                                                // wenn WLAN-Status "online"
    if (!ntp_ok) {                                                    // wenn NTP-Dienst noch nicht gestartet
      Serial.println(F("Synchronisierung wird gestartet"));
      configTime(time_zone_p, time_ntp1_p, time_ntp2_p);              // NTP-Zeitsynchronisierung starten
      ntp_ok = true;                                                  // NTP-Dienst wurde gestartet
    }

    if (!mqtt_ok) {                                                   // wenn MQTT-Dienst noch nicht gestartet
      if (mqtt_actv) {                                                // wenn MQTT aktiviert
        mqttClient.onMessage(onMqttMessage);                          // Callback für empfangene MQTT-Nachrichten setzen
        mqttClient.setServer(mqtt_addr_p, mqtt_port_i);               // MQTT-Serverdaten konfigurieren
        mqttClient.setKeepAlive(120);                                  // Verbindungs-Timeout 60 Sekunden
        if (mqtt_user_p && mqtt_pass_p) mqttClient.setCredentials(mqtt_user_p, mqtt_pass_p); // wenn Username und Passwort gesetzt -> Zugangsdaten konfigurieren
        if (mqtt_ltop_p) mqttClient.setWill(mqtt_ltop_p, 0, true, mqtt_lmsg_p);              // wenn LWT-Topic gesetzt -> LWT konfigurieren
      }
      mqtt_ok = true;                                                 // MQTT-Dienst wurde gestartet
    }
  }

  if ((wifistat == 2) && !mqttstat) {                                 // wenn WLAN im Online-Zustand und keine MQTT-Verbindung
    if (mqtttout == 255) {                                            // wenn Timeout-Zähler inaktiv
      if (mqtt_actv && !reboot) {                                     // wenn MQTT aktiviert und kein Reboot angefordert wurde
        mqttClient.connect();                                         // MQTT-Verbindung herstellen
        mqtttout = MQTTTOUT;                                          // MQTT-Timeout-Zähler starten
        Serial.println(F("MQTT-Verbindungsaufbau"));
      }
    }
  }

  if (mqtttout == 0) {                                                // wenn MQTT-Timeout-Zähler abgelaufen
    mqtttout = 255;                                                   // Timeout-Zähler stoppen
    if (wifistat == 2 && !mqttstat) {                                 // wenn WLAN im Online-Zustand und keine MQTT-Verbindung
      if (mqttClient.connected()) {                                   // wenn MQTT-Verbindung hergestellt
        if (mqtt_ltop_p) mqttClient.publish(mqtt_ltop_p, 0, true, mqtt_smsg_p); // wenn LWT-Topic gesetzt -> LWT-Startnachricht senden
        mqttstat = true;                                              // MQTT-Status auf online setzen
        Serial.println(F("MQTT-Verbindung hergestellt"));
        for (i = 0; i < 11; i ++) {                                   // 11 Datenfelder mit MQTT-Topics bearbeiten
          if (sensorlist_p[i]) mqttClient.subscribe(sensorlist_p[i], 0); // wenn MQTT-Topic vorhanden -> abonnieren
        }
      }
      else {                                                          // wenn Verbindungsaufbau nicht erfolgreich
        Serial.println(F("MQTT-Fehler "));
      }
    }
  }

  if (mqttstat && !mqttClient.connected()) {                          // wenn MQTT-Status in den Offline-Status wechselt
    mqttstat = false;                                                 // neuen Status "offline" setzen
    Serial.println(F("MQTT-Verbindung unterbrochen"));
    mqtttout = MQTTTOUT;                                              // MQTT-Timeout-Zähler starten und neue Verbindung versuchen
  }

  // ------------------------------------------------------------------- Serielle Schnittstelle abfragen ------------------------------------------------------

  for (i = 0; i < Serial.available(); i ++) {                         // Schleife zum Abholen aller empfangenen Zeichen
    serialchar = Serial.read();                                       // empfangenes Zeichen holen
    if ((serialchar >= 32) && (serialstr.length() < 50)) {            // wenn kein Steuerzeichen und maximale Länge noch nicht erreicht
      serialstr += serialchar;                                        // empfangenes Zeichen an Puffer-String anhängen
      Serial.print(serialchar);                                       // empfangenes Zeichen als Echo wieder ausgeben
    }
    if (serialchar == '\n') {                                         // wenn empfangenes Zeichen Newline (ASCII 10)
      Serial.println();                                               // empfangenes Zeichen als Echo wieder ausgeben
      if (serialstr.length() > 0) {                                   // wenn zuvor Zeichen empfangen wurden
        serialcmd = serialstr.substring(0, 4);                        // Kommando vom Puffer-String holen
        serialcmd.toLowerCase();                                      // in Kleinbuchstaben wandeln
        if (serialcmd == "ssid") {                                    // wenn empfangenes Kommando "wifi"
          newssid = serialstr.substring(5);                           // der restliche String enthält die neue SSID
          Serial.println();
          Serial.print(F("SSID: "));
          Serial.println(newssid);                                    // eingegebene SSID zur Bestätigung ausgeben
        }
        if (serialcmd == "pass") {                                    // wenn empfangenes Kommando "pass"
          newpass = serialstr.substring(5);                           // der restliche String enthält das neue Passwort
          Serial.println();
          Serial.print(F("Pass: "));
          Serial.println(newpass);                                    // eingegebenes Passwort zur Bestätigung ausgeben
        }
        if (serialcmd == "list") {                                    // wenn empfangenes Kommando "list"
          Serial.println();
          Serial.print(F("SSID: "));
          if (newssid.length() > 0) Serial.println(newssid);          // wenn SSID eingegeben -> wieder ausgeben
          else Serial.println(F("<leer>"));
          Serial.print(F("Pass: "));
          if (newpass.length() > 0) Serial.println(newpass);          // wenn Passwort eingegeben -> wieder ausgeben
          else Serial.println(F("<leer>"));
        }
        if (serialcmd == "save") {                                    // wenn empfangenes Kommando "save"
          Serial.println();
          if (filestat) {                                             // wenn Dateisystem ok
            if ((newssid.length() > 0) && (newpass.length() > 0)) {   // wenn SSID und Passwort eingegeben
              filedata[0] = newssid;                                  // neue SSID in Dateipuffer legen
              filedata[1] = newpass;                                  // neues Passwort in Dateipuffer legen
              Serial.println();
              write_cfgfile(wifi_filename, 2);                        // WLAN-Zugangsdaten ins Dateisystem schreiben (2 Zeilen)
              Serial.println(F("Daten sind ok, Neustart"));
              if (filestat) LittleFS.remove(netw_filename);           // wenn Dateisystem vorhanden -> Netzwerkeinstellungen löschen
              boottout = BOOTTOUT;                                    // Neustart-Timer setzen
            }
            else Serial.println(F("Daten sind unvollstaendig"));
          }
          else Serial.println(F("Dateisystemfehler"));
        }
        serialstr = "";                                               // Puffer-String wieder löschen
      }
      else {                                                          // wenn keine Zeichen eingegeben wurden
        Serial.println(F("Kommandos:"));                              // Hilfetext ausgeben
        Serial.println(F("ssid SSID      (WLAN-Name eingeben)"));
        Serial.println(F("pass Passwort  (WLAN-Passwort eingeben)"));
        Serial.println(F("list           (WLAN-Daten zur Kontrolle anzeigen)"));
        Serial.println(F("save           (WLAN-Daten speichern und neu starten)"));
      }
    }
  }
}
